﻿using System.Collections.Generic;
using UnityEngine;

namespace KerbalMass
{
    /// <summary>
    /// This module adds and substracts the mass of it's parent part
    /// depending of the number of Kerbals on board.
    /// </summary>
    public class KerbalMassModule : PartModule
    {
        /// <summary>
        /// Current crew number in the part.
        /// </summary>
        [UI_FloatRange(controlEnabled = true, maxValue = 1f, minValue = 0f, scene = UI_Scene.Editor, stepIncrement = 1f)]
        [KSPField(guiActiveEditor = true, guiName = "Crew", isPersistant = false)]
        public float crewNumber = 0f;

        /// <summary>
        /// By which amount the part's mass has been changed through this module.
        /// </summary>
        [KSPField(isPersistant = true)]
        public float MassChangedBy = 0f;

        /// <summary>
        /// If it's the first time this module loads in the flight scene.
        /// </summary>
        [KSPField(isPersistant = true)]
        public bool firstFlight = true;

        /// <summary>
        /// This switch enables the slider in the editor for parts
        /// that are otherwise unaffected by this mod.
        /// </summary>
        [KSPField(isPersistant = true)]
        public bool isLimitedToEditor = false;

        /// <summary>
        /// This attribute is only accounted for if it is set to a value > 0 
        /// and isLimitedToEditor is set to true.
        /// It defines the crew capacity of this part.
        /// </summary>
        [KSPField(isPersistant = true)]
        public float editorCrewCapacity = -1f;


        /// <summary>
        /// Called once after the part is created or loaded.
        /// Perfect time to set up everything.
        /// </summary>
        public void Start()
        {
            // Update slider's max value
            if (HighLogic.LoadedSceneIsEditor)
            {
                UI_FloatRange slider = (UI_FloatRange) Fields["crewNumber"].uiControlEditor;
                if (isLimitedToEditor == false)
                    slider.maxValue = part.CrewCapacity;
                else
                    if (editorCrewCapacity > 0)
                        slider.maxValue = editorCrewCapacity;
                    else
                    {
                        // So the isLimitedToEditor is set to true but there's no
                        // useable value in editorCrewCapacity. Don't show the slider!
                        Fields["crewNumber"].guiActiveEditor = false;
                    }

                if (KerbalMassConfigManager.DefaultPartMassIncludesKerbalMass == true && isLimitedToEditor == false)
                    part.mass -= part.CrewCapacity * KerbalMassConfigManager.Kerbal_Mass;
            }

            // modify initial part mass
            if (HighLogic.LoadedSceneIsFlight && isLimitedToEditor == false)
            {
                MassChangedBy = part.protoModuleCrew.Count * KerbalMassConfigManager.Kerbal_Mass;

                if (firstFlight == true)
                {
                    // Because it's the first time this module is loaded in a flight scene,
                    // have to make some initial changes.

                    if (KerbalMassConfigManager.DefaultPartMassIncludesKerbalMass == true)
                        part.mass -= part.CrewCapacity * KerbalMassConfigManager.Kerbal_Mass;

                    part.mass += MassChangedBy;
                    firstFlight = false;
                }
            }
        }
        
        /// <summary>
        /// Updates the part mass depending on the number of Kerbals on board.
        /// </summary>
        public void FixedUpdate()
        {
            float currentCrew = 0f;
            float massChange = 0f;

            if (HighLogic.LoadedSceneIsEditor)
                currentCrew = crewNumber;

            if (HighLogic.LoadedSceneIsFlight && isLimitedToEditor == false)
                currentCrew = part.protoModuleCrew.Count;

            massChange = currentCrew * KerbalMassConfigManager.Kerbal_Mass;
            if (KerbalMassConfigManager.DefaultPartMassIncludesKerbalMass == true)
                massChange *= -1;

            if (MassChangedBy != massChange)
            {
                part.mass -= MassChangedBy; // revert old mass change
                MassChangedBy = massChange;
                part.mass += MassChangedBy; // apply new mass change

                Debug.Log("[KerbalMass] Part mass is now " + part.mass);
            }
        }
    }
}
